<?php
/**
 * Plugin Name: FormOMatic
 * Plugin URI: https://formomatic.com
 * Description: Embed beautiful, customized Google Forms on your WordPress site using FormOMatic.
 * Version: 1.0.0
 * Author: FormOMatic
 * Author URI: https://formomatic.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: formomatic
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

define('FORMOMATIC_VERSION', '1.0.0');
define('FORMOMATIC_APP_URL', 'https://formomatic.com/app');
define('FORMOMATIC_API_URL', 'https://formomatic.com/app/api');

/**
 * Register the [formomatic] shortcode
 */
function formomatic_shortcode($atts) {
    $atts = shortcode_atts([
        'id' => '',
    ], $atts, 'formomatic');

    $id = sanitize_text_field($atts['id']);

    if (empty($id)) {
        return '<!-- FormOMatic: No form ID provided -->';
    }

    // Enqueue styles and scripts properly
    wp_enqueue_style(
        'formomatic-embed-styles',
        FORMOMATIC_APP_URL . '/assets/css/styles-embed.css',
        [],
        FORMOMATIC_VERSION
    );

    wp_enqueue_script(
        'formomatic-customform',
        FORMOMATIC_APP_URL . '/assets/js/customform.js',
        [],
        FORMOMATIC_VERSION,
        true
    );

    // Enqueue form-specific embed script
    wp_enqueue_script(
        'formomatic-embed-' . $id,
        FORMOMATIC_APP_URL . '/api/embed.php?id=' . esc_attr($id),
        ['formomatic-customform'],
        FORMOMATIC_VERSION,
        true
    );

    // Generate unique container ID
    $container_id = 'formomatic-' . esc_attr($id);

    return sprintf('<div id="%s"></div>', $container_id);
}
add_shortcode('formomatic', 'formomatic_shortcode');

/**
 * Register settings
 */
function formomatic_register_settings() {
    register_setting('formomatic_settings', 'formomatic_api_key', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
}
add_action('admin_init', 'formomatic_register_settings');

/**
 * Fetch forms from API
 */
function formomatic_get_forms() {
    $api_key = get_option('formomatic_api_key');

    if (empty($api_key)) {
        return [];
    }

    // Cache forms for 5 minutes
    $cache_key = 'formomatic_forms_' . md5($api_key);
    $cached = get_transient($cache_key);

    if ($cached !== false) {
        return $cached;
    }

    $response = wp_remote_get(FORMOMATIC_API_URL . '/forms-list.php?api_key=' . urlencode($api_key), [
        'timeout' => 10,
    ]);

    if (is_wp_error($response)) {
        return [];
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (!$data || !$data['success']) {
        return [];
    }

    $forms = $data['forms'] ?? [];

    // Cache for 5 minutes
    set_transient($cache_key, $forms, 5 * MINUTE_IN_SECONDS);

    return $forms;
}

/**
 * AJAX handler for fetching forms
 */
function formomatic_ajax_get_forms() {
    check_ajax_referer('formomatic_nonce', 'nonce');

    if (!current_user_can('edit_posts')) {
        wp_send_json_error('Unauthorized');
    }

    $forms = formomatic_get_forms();
    wp_send_json_success($forms);
}
add_action('wp_ajax_formomatic_get_forms', 'formomatic_ajax_get_forms');

/**
 * Register Gutenberg block
 */
function formomatic_register_block() {
    if (!function_exists('register_block_type')) {
        return;
    }

    wp_register_script(
        'formomatic-block',
        plugins_url('assets/block.js', __FILE__),
        ['wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'],
        FORMOMATIC_VERSION,
        true
    );

    // Pass data to the block script
    wp_localize_script('formomatic-block', 'formomaticBlock', [
        'apiKey' => get_option('formomatic_api_key'),
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('formomatic_nonce'),
    ]);

    register_block_type('formomatic/form', [
        'editor_script' => 'formomatic-block',
        'render_callback' => 'formomatic_block_render',
        'attributes' => [
            'formId' => [
                'type' => 'string',
                'default' => '',
            ],
        ],
    ]);
}
add_action('init', 'formomatic_register_block');

/**
 * Ensure block script is enqueued in the editor
 */
function formomatic_enqueue_block_editor_assets() {
    wp_enqueue_script('formomatic-block');
}
add_action('enqueue_block_editor_assets', 'formomatic_enqueue_block_editor_assets');

/**
 * Render the Gutenberg block on the frontend
 */
function formomatic_block_render($attributes) {
    $form_id = $attributes['formId'] ?? '';
    return formomatic_shortcode(['id' => $form_id]);
}

/**
 * Add admin menu
 */
function formomatic_admin_menu() {
    add_menu_page(
        'FormOMatic',
        'FormOMatic',
        'manage_options',
        'formomatic',
        'formomatic_admin_page',
        'dashicons-feedback',
        30
    );
}
add_action('admin_menu', 'formomatic_admin_menu');

/**
 * Admin page content
 */
function formomatic_admin_page() {
    $api_key = get_option('formomatic_api_key');
    $forms = [];
    $api_error = false;

    if (!empty($api_key)) {
        $forms = formomatic_get_forms();
        if (empty($forms)) {
            // Check if it's an API error or just no forms
            $response = wp_remote_get(FORMOMATIC_API_URL . '/forms-list.php?api_key=' . urlencode($api_key), ['timeout' => 10]);
            if (!is_wp_error($response)) {
                $body = json_decode(wp_remote_retrieve_body($response), true);
                if (isset($body['error'])) {
                    $api_error = $body['error'];
                }
            }
        }
    }

    // Handle form submission
    if (isset($_POST['formomatic_save_settings']) && check_admin_referer('formomatic_settings_nonce')) {
        $new_key = isset($_POST['formomatic_api_key']) ? sanitize_text_field(wp_unslash($_POST['formomatic_api_key'])) : '';
        update_option('formomatic_api_key', $new_key);

        // Clear cache
        delete_transient('formomatic_forms_' . md5($api_key));
        delete_transient('formomatic_forms_' . md5($new_key));

        $api_key = $new_key;

        echo '<div class="notice notice-success"><p>Settings saved!</p></div>';

        // Refresh forms
        $forms = formomatic_get_forms();
    }
    ?>
    <div class="wrap">
        <h1>FormOMatic</h1>

        <!-- Settings Panel -->
        <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); max-width: 600px; margin-bottom: 20px;">
            <h2 style="margin-top: 0;">Settings</h2>

            <form method="post">
                <?php wp_nonce_field('formomatic_settings_nonce'); ?>

                <table class="form-table" style="margin: 0;">
                    <tr>
                        <th scope="row" style="padding-left: 0;"><label for="formomatic_api_key">API Key</label></th>
                        <td>
                            <input type="text" id="formomatic_api_key" name="formomatic_api_key"
                                   value="<?php echo esc_attr($api_key); ?>"
                                   class="regular-text" placeholder="fm_xxxxxxxxxxxxxxxx">
                            <p class="description">
                                Get your API key from <a href="https://formomatic.com/app/account.php" target="_blank">your FormOMatic account</a>.
                            </p>
                        </td>
                    </tr>
                </table>

                <p style="margin-bottom: 0;">
                    <input type="submit" name="formomatic_save_settings" class="button button-primary" value="Save Settings">
                </p>
            </form>

            <?php if ($api_error): ?>
            <div class="notice notice-error inline" style="margin-top: 10px;">
                <p><?php echo esc_html($api_error); ?></p>
            </div>
            <?php endif; ?>
        </div>

        <!-- Forms Panel -->
        <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); max-width: 800px;">
                <h2 style="margin-top: 0;">Your Forms</h2>

                <?php if (empty($api_key)): ?>
                <p style="color: #666;">Enter your API key to see your forms.</p>

                <?php elseif (empty($forms)): ?>
                <p style="color: #666;">No forms found. <a href="https://formomatic.com/app/builder.php" target="_blank">Create your first form</a>.</p>

                <?php else: ?>
                <table class="wp-list-table widefat striped">
                    <thead>
                        <tr>
                            <th>Form Name</th>
                            <th>Shortcode</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($forms as $form): ?>
                        <tr>
                            <td><strong><?php echo esc_html($form['name']); ?></strong></td>
                            <td>
                                <code id="shortcode-<?php echo esc_attr($form['id']); ?>">[formomatic id="<?php echo esc_attr($form['id']); ?>"]</code>
                            </td>
                            <td>
                                <button type="button" class="button button-small" onclick="copyShortcode('<?php echo esc_attr($form['id']); ?>')">
                                    Copy
                                </button>
                                <a href="https://formomatic.com/app/builder.php?id=<?php echo esc_attr($form['id']); ?>" target="_blank" class="button button-small">
                                    Edit
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php endif; ?>

                <p style="margin-top: 20px;">
                    <a href="https://formomatic.com/app/builder.php" target="_blank" class="button button-primary">
                        Create New Form
                    </a>
                    <a href="https://formomatic.com/app/" target="_blank" class="button">
                        Open Dashboard
                    </a>
                </p>
        </div>

        <!-- Help Section -->
        <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); margin-top: 20px; max-width: 800px;">
            <h2 style="margin-top: 0;">How to Embed Forms</h2>

            <h3>Option 1: Shortcode</h3>
            <p>Copy the shortcode from the table above and paste it into any post or page.</p>
            <code style="display: block; background: #f5f5f5; padding: 10px; border-radius: 4px;">[formomatic id="your-form-id"]</code>

            <h3 style="margin-top: 20px;">Option 2: Gutenberg Block</h3>
            <p>Search for "FormOMatic" in the block inserter. Select your form from the dropdown in the block settings.</p>
        </div>
    </div>

    <script>
    function copyShortcode(formId) {
        var shortcode = document.getElementById('shortcode-' + formId).textContent;
        navigator.clipboard.writeText(shortcode).then(function() {
            alert('Shortcode copied!');
        });
    }
    </script>
    <?php
}
